/*
Half-Life MAP viewing utility.
Copyright (C) 2003  Ryan Samuel Gregg

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#pragma once
#include "stdafx.h"
#include "WorldObject.h"
#include "Brush.h"
#include "Entity.h"
#include "Frustum.h"
#include "TextureManager.h"

__sealed __gc class CWorld : public CWorldObject
{
private:
	int iBrushCount;
	ArrayList *Entities;

public:
	CWorld(CConfig *Config) : CWorldObject(Config)
	{
		Entities = new ArrayList();
	}

	void AddEntity(CEntity *Entity)
	{
		Entities->Add(Entity);
	}

	ArrayList *GetEntities()
	{
		return Entities;
	}

	int GetBrushCount()
	{
		return iBrushCount;
	}

	bool GetWADString(String **sWADString)
	{
		if(Entities->Count == 0)
			return false;

		ArrayList *ArgVals;
		CArgVal *ArgVal;
		ArgVals = static_cast<CEntity*>(Entities->get_Item(0))->GetArgVals();

		for(int i = 0; i < ArgVals->Count; i++)
		{
			ArgVal = static_cast<CArgVal*>(ArgVals->get_Item(i));
			if(ArgVal->GetArg()->ToLower()->Equals("wad"))
			{
				*sWADString = ArgVal->GetVal();
				return true;
			}
		}

		return false;
	}

	void UpdateBrushCount()
	{
		iBrushCount = 0;

		for(int i = 0; i < Entities->Count; i++)
		{
			iBrushCount += static_cast<CEntity*>(Entities->get_Item(i))->GetBrushCount();
		}
	}

	void UpdateBoundingBoxes()
	{
		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->UpdateBoundingBoxes();
		}
	}

	void UpdateColors()
	{
		Random *Rnd = new Random();
		CEntity *Entity;

		for(int i = 0; i < Entities->Count; i++)
		{
			Entity = static_cast<CEntity*>(Entities->get_Item(i));
			Entity->SetColor(Rnd->Next(256), Rnd->Next(256), Rnd->Next(256));

			for(int j = 0; j < Entity->GetBrushes()->Count; j++)
			{
				static_cast<CBrush*>(Entity->GetBrushes()->get_Item(j))->SetColor(Rnd->Next(256), Rnd->Next(256), Rnd->Next(256));
			}
		}
	}

	void CullWorld(CFrustum *Frustum)
	{
		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->CullEntity(Frustum);
		}
	}

	void UpdateTexCoords(CTextureManager *TextureManager)
	{
		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->UpdateTexCoords(TextureManager);
		}
	}

	void DrawWorldTextured()
	{
		glEnable(GL_CULL_FACE);
		glEnable(GL_DEPTH_TEST);
		glEnable(GL_TEXTURE_2D);
		glColor3ub(255, 255, 255);

		if(Config->bLightScene)
		{
			glEnable(GL_LIGHTING);
			glEnable(GL_LIGHT0);
			glEnable(GL_LIGHT1);
		}

		if(Config->bFog)
		{
			glEnable(GL_FOG);
		}

		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->DrawEntityTextured();
		}

		if(Config->bFog)
		{
			glDisable(GL_FOG);
		}

		if(Config->bLightScene)
		{
			glDisable(GL_LIGHT0);
			glDisable(GL_LIGHT1);
			glDisable(GL_LIGHTING);
		}
	}

	void DrawWorldSolid()
	{
		glEnable(GL_CULL_FACE);
		glEnable(GL_DEPTH_TEST);
		glDisable(GL_TEXTURE_2D);

		if(Config->bLightScene)
		{
			glEnable(GL_LIGHTING);
			glEnable(GL_LIGHT0);
			glEnable(GL_LIGHT1);
		}

		if(Config->bFog)
		{
			glEnable(GL_FOG);
		}

		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->DrawEntitySolid();
		}

		if(Config->bFog)
		{
			glDisable(GL_FOG);
		}

		if(Config->bLightScene)
		{
			glDisable(GL_LIGHT0);
			glDisable(GL_LIGHT1);
			glDisable(GL_LIGHTING);
		}
	}

	void DrawWorldWireFrame()
	{
		glDisable(GL_CULL_FACE);
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_TEXTURE_2D);

		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->DrawEntityWireFrame();
		}
	}

	void DrawWorldPoints()
	{
		glDisable(GL_CULL_FACE);
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_TEXTURE_2D);

		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->DrawEntityPoints();
		}
	}

	void Highlight(bool bPrepared)
	{
		if(!bPrepared)
		{
			PrepareHighlight();
		}

		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->Highlight(true);
		}
	}

	void Outline()
	{
		glEnable(GL_CULL_FACE);
		glEnable(GL_DEPTH_TEST);
		glDisable(GL_TEXTURE_2D);
		
		glColor4f(Config->cOutlineColor.R, Config->cOutlineColor.G, Config->cOutlineColor.B, Config->cOutlineColor.A);

		glDepthFunc(GL_LEQUAL);

		for(int i = 0; i < Entities->Count; i++)
		{
			static_cast<CEntity*>(Entities->get_Item(i))->Outline();
		}

		glDepthFunc(GL_LESS);
	}
};